//--------------------------------------------------------------------------------------
// File:		SurfaceExtensions.h.
// Namespace:	Global.
// Description:	
// Author:		Grant Davies.
// Platform:	ALL.
// 
//--------------------------------------------------------------------------------------

#ifndef __SURFACEEXTENSIONS__
#define __SURFACEEXTENSIONS__


//--------------------------------------------------------------------------------------
// Includes.
//--------------------------------------------------------------------------------------

#include <il\il_wrap.h>

#ifndef __TYPES__
#include "Types.h"
#endif //__TYPES__

#ifndef __COLOUR__
#include "Colour.h"
#endif //__COLOUR__


//--------------------------------------------------------------------------------------
// Constants.
//--------------------------------------------------------------------------------------



//--------------------------------------------------------------------------------------
// Data Types.
//--------------------------------------------------------------------------------------


//--------------------------------------------------------------------------------------
// Function Definitions.
//--------------------------------------------------------------------------------------

//--------------------------------------------------------------------------------------
// Description:	Get the pitch of the specified surface.
// Parameters:	The surface to get the pitch of.
// Returns:		The pitch of the specified surface.
// Notes:		The pitch is the number of BYTES (not pixels) that a single scanline in
//				the surface occupies.
//--------------------------------------------------------------------------------------
static inline int getSurfacePitch(ilImage* surface)
{
	return surface->Width() * surface->Bpp();
}

//--------------------------------------------------------------------------------------
// Description:	Blit function from one surface to the other.
// Parameters:	The surface to blit from,
//				The portion of the surface to blit from,
//				The surface to blit to,
//				The location on the surface to blit to (only x and y are used, not w and
//				h).
// Returns:		Zero on success; non-zero on error.
//--------------------------------------------------------------------------------------
static inline int blitSurface
(
	ilImage* srcSurface,
	Rect* srcRect,
	ilImage* dstSurface,
	Rect* dstRect
)
{
	// Provide a source rect if one isn't already specified.
	Rect tmpSrcRect;
	if (!srcRect)
	{
		srcRect = &tmpSrcRect;
		srcRect->x = 0;
		srcRect->y = 0;
		srcRect->w = srcSurface->Width();
		srcRect->h = srcSurface->Height();
	}

	// Provide a destination rect if one isn't already specified.
	Rect tmpDstRect;
	if (!dstRect)
	{
		dstRect = &tmpDstRect;
		dstRect->x = 0;
		dstRect->y = 0;
	}

	// Ensure the dimensions and positions of the rects are valid.

	int dstSurfaceWidth = (int)dstSurface->Width();
	int dstSurfaceHeight = (int)dstSurface->Height();

	if (dstRect->x + srcRect->w >= dstSurfaceWidth)
	{
		srcRect->w = dstSurfaceWidth - dstRect->x;
	}

	if (dstRect->y + srcRect->h >= dstSurfaceHeight)
	{
		srcRect->y = dstSurfaceHeight - dstRect->y;
	}

	if (dstRect->x < 0)
	{
		srcRect->w += dstRect->x;
		dstRect->x = 0;
	}

	if (dstRect->y < 0)
	{
		srcRect->h += dstRect->y;
		dstRect->y = 0;
	}

	if (srcRect->w < 0)
	{
		return 0;
	}

	if (srcRect->h < 0)
	{
		return 0;
	}

	if (dstRect->x + srcRect->w > dstSurfaceWidth)
	{
		srcRect->w = dstSurfaceWidth - dstRect->x;
	}

	if (dstRect->y + srcRect->h > dstSurfaceHeight)
	{
		srcRect->h = dstSurfaceHeight - dstRect->y;
	}

	const byte* srcPixels = static_cast<const byte*>(srcSurface->GetData());
	byte* dstPixels = static_cast<byte*>(dstSurface->GetData());

	// Copy across the pixels.
	for (int y = 0; y < srcRect->h; y++)
	{
		int dstY = dstRect->y + y;

		const int* srcPixelsLine = (const int*)(&srcPixels[y * getSurfacePitch(srcSurface)]);
		int* dstPixelsLine = (int*)(&dstPixels[dstY * getSurfacePitch(dstSurface)]);

		for (int x = 0; x < srcRect->w; x++)
		{
			int dstX = dstRect->x + x;

			dstPixelsLine[dstX] = srcPixelsLine[x];
		}
	}

	return 0;
}

//--------------------------------------------------------------------------------------
// Description:	Fill a surface rectangle with a colour.
// Parameters:	The surface to fill,
//				The rectangle to fill,
//				The colour to fill with.
// Returns:		None.
//--------------------------------------------------------------------------------------
static inline void fillRect(ilImage* surface, Rect& rect, Colour colour)
{
	Colour* surfacePixels = (Colour*)(surface->GetData());

	surfacePixels += rect.y * surface->Width() + rect.x;
	for (int y = 0; y < rect.h; y++)
	{
		for (int x = 0; x < rect.w; x++)
		{
			*surfacePixels = colour;
			surfacePixels++;
		}

		surfacePixels += surface->Width() - rect.w;
	}
}

//--------------------------------------------------------------------------------------
// Description:	Create a surface of the specified dimensions.
// Parameters:	The width of the surface,
//				The height of the surface.
// Returns:		A pointer to the new surface.
//--------------------------------------------------------------------------------------
static inline ilImage* createSurface(int width, int height)
{
	assert(width > 0);
	assert(height > 0);

	ilImage* surface = new ilImage;

	surface->Resize(width, height, 1);
	ILboolean result = surface->Convert(IL_RGBA);

	int bpp = surface->Bpp();
	assert(4 == bpp);

	// Fill the surface with the background colour.
	fillRect(surface, Rect(0, 0, width, height), Colour(0, 0, 0));

	return surface;
}

//--------------------------------------------------------------------------------------
// Description:	
// Parameters:	
// Returns:		
//--------------------------------------------------------------------------------------
static inline void writeSurfaceToFile(ilImage* surface, std::string fileName)
{
	// Because surfaces are stored upside down in devil image library, need to flip it
	// before writing.
	ilImage* tmpSurface = new ilImage(*surface);
	tmpSurface->Flip();

	// Delete the existing file first.
	::unlink(fileName.c_str());

	// Because ilImage uses char* instead of const char*, need to copy the file name
	// string.
	char* tmpFileName = ::strdup(fileName.c_str());
	tmpSurface->Save(tmpFileName);
	::free(tmpFileName);

	// Delete the temporary surface.
	deleteAndClear(tmpSurface);
}

#endif //__SURFACEEXTENSIONS__
