//--------------------------------------------------------------------------------------
// File:		FontExporter.h.
// Namespace:	Global.
// Description:	The font exporter application.
// Author:		Grant Davies.
// Platform:	ALL.
// 
//--------------------------------------------------------------------------------------

#ifndef __FONTEXPORTER__
#define __FONTEXPORTER__


//--------------------------------------------------------------------------------------
// Includes.
//--------------------------------------------------------------------------------------

#include <string>
#include <map>
#include <vector>

#ifndef __FONTENGINE__
#include "FontEngine.h"
#endif //__FONTENGINE__

#ifndef __FONT__
#include "Font.h"
#endif //__FONT__

#ifndef __DOOM3GLYPHDESCRIPTOR__
#include "Doom3GlyphDescriptor.h"
#endif //__DOOM3GLYPHDESCRIPTOR__

#ifndef __GLYPHPAGE__
#include "GlyphPage.h"
#endif //__GLYPHPAGE__

#ifndef __DESCRIPTORFILE__
#include "DescriptorFile.h"
#endif //__DESCRIPTORFILE__


//--------------------------------------------------------------------------------------
// Constants.
//--------------------------------------------------------------------------------------

#define CLASS_NAME FontExporter


//--------------------------------------------------------------------------------------
// Data Types.
//--------------------------------------------------------------------------------------

//--------------------------------------------------------------------------------------
// Type name:	FontExporter.
// Description:	The font exporter application.
//--------------------------------------------------------------------------------------

class FontExporter
{
public:

	//--------------------------------------------------------------------------------------
	// Public Member Functions.
	//--------------------------------------------------------------------------------------

	FontExporter();
	~FontExporter();

	bool export();
	bool export(int fontSize, float glyphScale);
	bool initialise(int argumentCount, char* arguments[]);

private:

	//--------------------------------------------------------------------------------------
	// Private Member Functions.
	//--------------------------------------------------------------------------------------

	bool parseArguments(int argumentCount, char* arguments[]);
	void outputUsage() const;
	bool exportCharacter(int characterCode);
	bool checkCharacters();
	void configureGlyphDescriptor(
		Doom3GlyphDescriptor& descriptor,
		const Glyph& glyph,
		const Rect& rect,
		const GlyphPage& glyphPage);

	//--------------------------------------------------------------------------------------
	// Private Data Members.
	//--------------------------------------------------------------------------------------

	// The font engine.
	FontEngine fontEngine;

	// The font to export.
	Font* font;

	// The name of the font file.
	std::string fontFileName;

	// The sizes of the fonts to export.
	typedef std::vector<int> IntVector;
	IntVector fontPointSizes;

	// The output folder that all output files are written into.
	std::string outputFolderName;

	// A mapping from glyph index to character code, which is useful for knowing if a
	// glyph index has already been exported (in the case of several character codes
	// mapping to a single glyph index - which is very common).  This saves a lot of
	// space on the exported texture.
	std::map<int, int> glyphIndicesToCharacterCodes;

	// A caching of all the exported glyphs, to be used when two different characters
	// map to the same glyph.
	Doom3GlyphDescriptor doom3GlyphDescriptors[Font::numCharactersToExport];

	// The texture pages containing the packed and exported glyphs.
	std::vector<GlyphPage> glyphPages;

	// The descriptor file contains information about each glpyh as well as pointing
	// to the portion of the texture containing the image data.
	DescriptorFile descriptorFile;

	// Whether or not x padding is being used to account for the negative x offset issue.
	Glyph::XOffsetFixType xOffsetFixType;

	// The output formats of the textures.
	std::vector<std::string> textureFormats;

	// Do not output any warnings about the X offsets.
	bool noXOffsetWarnings;

	// The title of this font.
	std::string fontTitle;
};


//--------------------------------------------------------------------------------------
// Function Definitions.
//--------------------------------------------------------------------------------------


#undef CLASS_NAME


#endif //__FONTEXPORTER__
